(* runtime configuration *)
(* solyga@gmx.de, 2025-09-21 *)
IMPLEMENTATION MODULE Config;

IMPORT Str, IO, FIO, Lib2, Lib3, Args, SO, Cfg;

CONST
  prg = Cfg.prg;
  ver = Cfg.ver;
  ext = Cfg.ext;
  wid = Cfg.wid;


PROCEDURE WrInfo( VAR b: SO.TBuffer );
BEGIN
  SO.WrStr( b, prg );
  SO.WrStr( b, ' v' );
  SO.WrCard( b, ver, 0 );
  SO.WrStr( b, ' (' );
  SO.WrStr( b, ext );
  SO.WrStr( b, '): Constant expression evaluator.' );
END WrInfo;

PROCEDURE WrAuth( VAR b: SO.TBuffer );
BEGIN
  SO.WrStr( b, 'Author: Steffen Solyga <solyga@gmx.de>' );
END WrAuth;

PROCEDURE WrWdth( VAR b: SO.TBuffer );
BEGIN
  SO.WrStr( b, 'Internal mantissa width: ' );
  SO.WrCard( b, Cfg.wid * 8, 1 );
  SO.WrStr( b, ' bits' );
END WrWdth;


PROCEDURE ParseArgs();

  PROCEDURE Help();
  VAR
    b: SO.TBuffer;
    s: ARRAY [0..127] OF CHAR;
  BEGIN
    SO.Init( b, s );
    SO.Clear( b ); SO.WrStr( b, "(* " ); WrInfo( b ); SO.WrStr( b, " *)" );
    IO.WrStr( s ); IO.WrLn();
    SO.Clear( b ); SO.WrStr( b, "(* " ); WrAuth( b ); SO.WrStr( b, " *)" );
    IO.WrStr( s ); IO.WrLn();
    SO.Clear( b ); SO.WrStr( b, "(* " ); WrWdth( b ); SO.WrStr( b, " *)" );
    IO.WrStr( s ); IO.WrLn();
    SO.Clear( b ); SO.WrStr( b, "syntax = '" ); SO.WrStr( b, prg ); SO.WrStr( b, "' { option } type { option } expression ." );
    IO.WrStr( s ); IO.WrLn();
    SO.Clear( b ); SO.WrStr( b, "type = 'r' | 'i' | 's' . (* reals, ints, sets *)" );
    IO.WrStr( s ); IO.WrLn();
    SO.Clear( b ); SO.WrStr( b, "option = '-h'            (* show this info and exit *)" );
    IO.WrStr( s ); IO.WrLn();
    SO.Clear( b ); SO.WrStr( b, "       | '-d'            (* show debug messages *)" );
    IO.WrStr( s ); IO.WrLn();
    SO.Clear( b ); SO.WrStr( b, "       | '-p' num        (* set output precision to num (reals) *)" );
    IO.WrStr( s ); IO.WrLn();
    SO.Clear( b ); SO.WrStr( b, "       | '-b' num        (* set input and output base to num (ints, sets) *)" );
    IO.WrStr( s ); IO.WrLn();
    SO.Clear( b ); SO.WrStr( b, "       | '-i' num        (* set input base to num (ints, sets) *)" );
    IO.WrStr( s ); IO.WrLn();
    SO.Clear( b ); SO.WrStr( b, "       | '-o' num        (* set output base to num (ints, sets) *)" );
    IO.WrStr( s ); IO.WrLn();
    SO.Clear( b ); SO.WrStr( b, "       | '-f' idf        (* set division flavor (ints) *)" );
    IO.WrStr( s ); IO.WrLn();
    SO.Clear( b ); SO.WrStr( b, "       | '-v'            (* be verbose *) ." );
    IO.WrStr( s ); IO.WrLn();
    SO.Clear( b ); SO.WrStr( b, "idf = 't' | 'f' | 'e' | 'c' | 'n' . (* trunc, floor, euclid, ceil, nearest *)" );
    IO.WrStr( s ); IO.WrLn();
    Lib3.SetReturnCode( 0 );
    HALT;
  END Help;

  PROCEDURE Error( t-: ARRAY OF CHAR );
  VAR
    b: SO.TBuffer;
    s: ARRAY [0..127] OF CHAR;
  BEGIN
    SO.Init( b, s );
    SO.WrStr( b, 'Error: ' );
    SO.WrStr( b, t );
    IF Args.len > 0 THEN
      SO.WrStr( b, ": '" );
      SO.WrStr( b, Args.str );
      SO.WrStr( b, "'" );
    END;
    SO.WrLn( b );
    SO.WrStr( b, "Try '" );
    SO.WrStr( b, prg );
    SO.WrStr( b, " -h' for help." );
    Lib3.FatalError( s );
  END Error;

  PROCEDURE DecNum( VAR x: CARDINAL );
  VAR
    ok: BOOLEAN;
  BEGIN
    x := Str.StrToCard( Args.str, 10, ok );
    IF ~ ok THEN Error( 'Invalid decimal number' ) END;
    Args.Read();
  END DecNum;

  PROCEDURE Type();
  BEGIN
    CASE CAP( Args.str[0] ) OF
    | 'R': data.typ := 'R';
    | 'I': data.typ := 'I';
    | 'S': data.typ := 'S';
    ELSE
      Error( 'Invalid type' );
    END;
    Args.Read();
  END Type;

  PROCEDURE Flavor( VAR x: CHAR );

    PROCEDURE Ok( c: CHAR ): BOOLEAN;
    CONST s = 'TFDECURN';
    VAR i: CARDINAL;
    BEGIN
      FOR i := 0 TO SIZE(s) - 1 DO
        IF c = s[i] THEN RETURN TRUE END;
      END;
      RETURN FALSE;
    END Ok;

  BEGIN
    x := CAP( Args.str[0] );
    IF ~ Ok( x ) THEN Error( 'Invalid division flavor' ) END;
    Args.Read();
  END Flavor;

  PROCEDURE Base( VAR x: CARDINAL );
  CONST
    bMin = 2;
    bMax = 10 + ( ORD('Z') + 1 - ORD('A') );  (* 36 *)
  VAR
    ok: BOOLEAN;
  BEGIN
    x := Str.StrToCard( Args.str, 10, ok );
    IF ~ ok OR ( x < bMin ) OR ( x > bMax ) THEN Error( 'Invalid base' ) END;
    Args.Read();
  END Base;

  PROCEDURE Precision( VAR x: CARDINAL );
  CONST
    min = 1;
    max = 256;
  VAR
    ok: BOOLEAN;
  BEGIN
    x := Str.StrToCard( Args.str, 10, ok );
    IF ~ ok OR ( x < min ) OR ( x > max ) THEN Error( 'Invalid precision' ) END;
    Args.Read();
  END Precision;

  PROCEDURE Options();
  VAR
    c: CHAR;
  BEGIN
    LOOP
      (* single char options only *)
      IF Args.len <= 1 THEN EXIT END;
      c := Args.str[ 0 ];
      IF ( c # '-' ) & ( c # '/' ) THEN EXIT END;
      IF Args.len > 2 THEN EXIT END;
      c := CAP( Args.str[1] );
      CASE c OF
      | '?', 'H': Help();
      | 'D'     : data.dbg := TRUE; Args.Read();
      | 'V'     : data.vrb := TRUE; Args.Read();
      | 'F'     : Args.Read(); Flavor( data.idf );
      | 'I'     : Args.Read(); Base( data.bi );
      | 'O'     : Args.Read(); Base( data.bo );
      | 'B'     : Args.Read(); Base( data.bi ); data.bo := data.bi;
      | 'P'     : Args.Read(); Precision( data.prc );
      ELSE
        EXIT;
      END;
    END;
  END Options;

  PROCEDURE String();
  VAR
    b: SO.TBuffer;
  BEGIN
    SO.Init( b, data.str );
    WHILE Args.idx <= Args.cnt DO
      SO.WrStr( b, Args.str );
      IF Args.idx < Args.cnt THEN SO.WrChar( b, ' ' ) END;
      Args.Read();
    END;
  END String;

BEGIN
  Args.Reset();
  Str.Copy( data.cmd, Args.str );
  Args.Read();
  Options();
  Type();
  Options();
  String();
END ParseArgs;

PROCEDURE InitDir();
VAR
  drv: SHORTCARD;
BEGIN
  drv := FIO.GetDrive();
  FIO.GetDir( drv, data.dir );
  Str.Prepend( data.dir, ':' );
  Str.Prepend( data.dir, CHR( ORD( 'A' ) + CARDINAL(drv-1) ) );
END InitDir;

PROCEDURE Show();
VAR
  b: SO.TBuffer;
  s: ARRAY [0..127] OF CHAR;

  PROCEDURE Out();
  BEGIN
    (* must be 2x IO (not FIO) for redirected stream beeing correct *)
    IO.WrStr( s );
    IO.WrLn();
  END Out;

  PROCEDURE ShowHdr();
  BEGIN
    SO.Clear( b ); WrInfo( b ); Out();
    SO.Clear( b ); WrAuth( b ); Out();
    (*
    SO.Clear( b ); WrWdth( b ); Out();
    *)
    SO.Clear( b ); SO.WrStr( b, 'Config.data:' ); Out();
  END ShowHdr;

  PROCEDURE Begin( n-: ARRAY OF CHAR );
  CONST w = 3;
  BEGIN
    SO.Clear( b ); SO.WrStr( b, '  ' ); SO.WrStrAdj( b, n, -w ); SO.WrChar( b, ' ' );
  END Begin;

  PROCEDURE ShowStr( n-: ARRAY OF CHAR; v-: ARRAY OF CHAR );
  BEGIN
    Begin( n ); SO.WrChar( b, "'" ); SO.WrStr( b, v ); SO.WrChar( b, "'" ); Out();
  END ShowStr;

  PROCEDURE ShowCard( n-: ARRAY OF CHAR; v: CARDINAL );
  CONST w = 1;
  BEGIN
    Begin( n ); SO.WrLngCard( b, v, w ); Out();
  END ShowCard;

  PROCEDURE ShowChar( n-: ARRAY OF CHAR; c: CHAR );
  BEGIN
    Begin( n ); SO.WrChar( b, c ); Out();
  END ShowChar;

BEGIN
  IF ~ data.vrb THEN RETURN END;
  SO.Init( b, s );
  ShowHdr();
  ShowStr(  'prg', data.prg );
  ShowCard( 'ver', data.ver );
  ShowStr(  'ext', data.ext );
  ShowStr(  'cmd', data.cmd );
  ShowStr(  'dir', data.dir );
  ShowCard( 'dbg', ORD( data.dbg ) );
  ShowCard( 'vrb', ORD( data.vrb ) );
  ShowChar( 'idf', data.idf );
  ShowChar( 'typ', data.typ );
  ShowCard( 'wid', data.wid );
  ShowCard( 'prc', data.prc );
  ShowCard( 'bi ', data.bi );
  ShowCard( 'bo ', data.bo );
  ShowStr(  'str', data.str );
END Show;

PROCEDURE InitModule();
BEGIN
  Lib2.Fill( data, 0 );
  data.prg := prg;
  data.ver := ver;
  data.ext := ext;
  data.idf := 'T';
  data.typ := 'R';
  data.wid := wid;
  data.prc := 10;
  data.bi  := 10;
  data.bo  := 10;
  data.str := '';
  InitDir();
  ParseArgs();
  Show();
END InitModule;

BEGIN
  InitModule();
END Config.
