(* large cardinals, solyga@gmx.de, 2022-11-17 *)
IMPLEMENTATION MODULE LC;

FROM SYSTEM IMPORT BYTE, ADR;
IMPORT Types, Lib, Lib2;


PROCEDURE Fatal( s-: ARRAY OF CHAR );
BEGIN
  Lib.FatalError( s );
  HALT;
END Fatal;


TYPE
  TSht = Types.TShtCard;
  TCard = Types.TCard;
  TInt = Types.TInt;


<* PUSH *> <* COVERFLOW - *> <* CHECKRANGE - *>
PROCEDURE Inc0( VAR x: BYTE; y: BYTE; VAR c: BOOLEAN );
VAR
  z: RECORD
    CASE : CARDINAL OF
    | 0: v: TCard;
    | 1: l, h: TSht;
    END;
  END;
BEGIN
  z.v := TSht( x ) + TSht( y ) + ORD( c );
  x := z.l;
  c := z.h > 0;
END Inc0;
<* POP *>

<* PUSH *> <* IOVERFLOW - *> <* CHECKRANGE - *>
PROCEDURE Dec0( VAR x: BYTE; y: BYTE; VAR c: BOOLEAN );
VAR
  z: RECORD
    CASE : CARDINAL OF
    | 0: v: TInt;
    | 1: l, h: TSht;
    END;
  END;
BEGIN
  z.v := INTEGER( TSht( x ) ) - TSht( y ) - TSht( ORD( c ) );
  x := z.l;
  c := z.h > 0;
END Dec0;
<* POP *>

PROCEDURE Cmp0( x, y: BYTE ): INTEGER;
BEGIN
  IF TSht( x ) < TSht( y ) THEN RETURN -1 ELSIF TSht( x ) > TSht( y ) THEN RETURN 1 END;
  RETURN 0;
END Cmp0;


<* PUSH *> <* CHECKINDEX - *> <* COVERFLOW - *>
PROCEDURE Len( x-: ARRAY OF BYTE ): CARDINAL;
VAR
  l: CARDINAL;
BEGIN
  l := SIZE( x );
  WHILE ( l > 1 ) & ( x[l-1] = BYTE(0) ) DO DEC( l ) END;
  RETURN l;
END Len;

PROCEDURE Len0( x-: ARRAY OF BYTE ): CARDINAL;
VAR
  l: CARDINAL;
BEGIN
  l := SIZE( x );
  WHILE ( l > 0 ) & ( x[l-1] = BYTE(0) ) DO DEC( l ) END;
  RETURN l;
END Len0;

PROCEDURE LenI( x-: ARRAY OF BYTE; i: CARDINAL ): CARDINAL;
VAR
  l: CARDINAL;
BEGIN
  l := SIZE( x );
  IF l <= i THEN RETURN 0 END;
  WHILE ( l > i ) & ( x[l-1] = BYTE(0) ) DO DEC( l ) END;
  RETURN l - i;
END LenI;
<* POP *>


<* PUSH *> <* CHECKINDEX - *> <* COVERFLOW - *>
PROCEDURE Cpy( VAR x: ARRAY OF BYTE; y-: ARRAY OF BYTE ): BOOLEAN;
VAR
  l, i: CARDINAL;
BEGIN
  l := Len( y );
  IF l > SIZE( x ) THEN RETURN FALSE END;
  FOR i := 0 TO l - 1 DO x[i] := y[i] END;
  WHILE l < SIZE( x ) DO x[l] := BYTE(0); INC( l ) END;
  RETURN TRUE;
END Cpy;

PROCEDURE Cmp( x-, y-: ARRAY OF BYTE ): INTEGER;
VAR
  lx, ly: CARDINAL;
  c: INTEGER;
BEGIN
  lx := Len( x ); ly := Len( y );
  IF lx < ly THEN RETURN -1 ELSIF lx > ly THEN RETURN 1 END;
  WHILE lx > 0 DO
    DEC( lx );
    c := Cmp0( x[lx], y[lx] );
    IF c # 0 THEN RETURN c END;
  END;
  RETURN 0;
END Cmp;
<* POP *>


<* PUSH *> <* CHECKINDEX - *> <* COVERFLOW - *>
PROCEDURE Inc( VAR x: ARRAY OF BYTE; y-: ARRAY OF BYTE ): BOOLEAN;
(* ADR( x ) = ADR( y ) ok *)
VAR
  i, l: CARDINAL;
  c: BOOLEAN;
BEGIN
  i := 0; l := Len0( y ); c := FALSE;
  WHILE i < l DO
    IF i >= SIZE( x ) THEN RETURN FALSE END;
    Inc0( x[i], y[i], c );
    INC( i );
  END;
  WHILE c DO
    IF i >= SIZE( x ) THEN RETURN FALSE END;
    Inc0( x[i], 0, c );
    INC( i );
  END;
  RETURN TRUE;
END Inc;

PROCEDURE Add( x-, y-: ARRAY OF BYTE; VAR z: ARRAY OF BYTE ): BOOLEAN;
BEGIN
  IF ADR( x ) = ADR( z ) THEN
    RETURN Inc( z, y );
  ELSIF ADR( y ) = ADR( z ) THEN
    RETURN Inc( z, x );
  ELSE
    RETURN Cpy( z, x ) & Inc( z, y );
  END;
END Add;
<* POP *>


<* PUSH *> <* CHECKINDEX - *> <* COVERFLOW - *>
PROCEDURE Dec( VAR x: ARRAY OF BYTE; y-: ARRAY OF BYTE ): BOOLEAN;
(* ADR( x ) = ADR( y ) ok *)
VAR
  i, l: CARDINAL;
  c: BOOLEAN;
BEGIN
  i := 0; l := Len0( y ); c := FALSE;
  WHILE i < l DO
    IF i >= SIZE( x ) THEN RETURN FALSE END;
    Dec0( x[i], y[i], c );
    INC( i );
  END;
  WHILE c DO
    IF i >= SIZE( x ) THEN RETURN FALSE END;
    Dec0( x[i], 0, c );
    INC( i );
  END;
  RETURN TRUE;
END Dec;

PROCEDURE Sub( x, y-: ARRAY OF BYTE; VAR z: ARRAY OF BYTE ): BOOLEAN;
BEGIN
  RETURN Dec( x, y ) & Cpy( z, x );
END Sub;
<* POP *>


<* PUSH *> <* CHECKINDEX - *> <* COVERFLOW - *> <* CHECKRANGE - *>
PROCEDURE Mul0( x, y: BYTE; VAR z, c: BYTE );
TYPE
  TCnvDesc = RECORD
    CASE : CARDINAL OF
    | 0: v: TCard;
    | 1: l, h: TSht;
    END;
  END;
VAR
  d: TCnvDesc;
BEGIN
  d.v := TSht( z ) + TSht( x ) * TSht( y ) + TSht( c );  (* FF + FF * FF + FF = FFFF *)
  z := d.l;
  c := d.h;
END Mul0;

PROCEDURE Mul( x-, y-: ARRAY OF BYTE; VAR z: ARRAY OF BYTE ): BOOLEAN;
VAR
  ix, lx, iy, ly, i: CARDINAL;
  c: TSht;
BEGIN
  (* ADR( x ) = ADR( y ) ok *)
  IF ( ADR( x ) = ADR( z ) ) OR ( ADR( y ) = ADR( z ) ) THEN RETURN FALSE END;
  lx := Len( x ); ly := Len( y );
  IF lx + ly - 1 > SIZE( z ) THEN RETURN FALSE END;
  Lib2.Fill( z, 0 );
  FOR iy := 0 TO ly - 1 DO
    c := 0;
    FOR ix := 0 TO lx - 1 DO
      i := ix + iy;
      Mul0( x[ix], y[iy], z[i], c );
    END;
    IF c > 0 THEN
      i := lx + iy;
      IF i >= SIZE( z ) THEN RETURN FALSE END;
      z[i] := c;
    END;
  END;
  RETURN TRUE;
END Mul;

PROCEDURE MulC( x, y-: ARRAY OF BYTE; VAR z: ARRAY OF BYTE ): BOOLEAN;
BEGIN
  RETURN Mul( x, y, z );
END MulC;

PROCEDURE Mpl( VAR x: ARRAY OF BYTE; y-: ARRAY OF BYTE ): BOOLEAN;
BEGIN
  RETURN MulC( x, y, x );
END Mpl;

PROCEDURE SqrC( x: ARRAY OF BYTE; VAR z: ARRAY OF BYTE ): BOOLEAN;
BEGIN
  RETURN Mul( x, x, z );
END SqrC;

PROCEDURE Sqr( VAR x: ARRAY OF BYTE ): BOOLEAN;
BEGIN
  RETURN SqrC( x, x );
END Sqr;
<* POP *>


<* PUSH *> <* CHECKINDEX - *> <* COVERFLOW - *> <* CHECKRANGE - *>
PROCEDURE Div( x, y-: ARRAY OF BYTE; VAR z, r: ARRAY OF BYTE ): BOOLEAN;

  PROCEDURE Cmp( x-: ARRAY OF BYTE; ix: CARDINAL; y-: ARRAY OF BYTE ): INTEGER;
  VAR
    lx, ly: CARDINAL;
    c: INTEGER;
  BEGIN
    lx := LenI( x, ix ); ly := Len0( y );
    IF lx < ly THEN RETURN -1 ELSIF lx > ly THEN RETURN 1 END;
    WHILE lx > 0 DO
      DEC( lx );
      c := Cmp0( x[ix+lx], y[lx] );
      IF c # 0 THEN RETURN c END;
    END;
    RETURN 0;
  END Cmp;

  (* PRE: Cmp(x,ix,y) >= 0  (required for x[ix] below) *)
  PROCEDURE Dec( VAR x: ARRAY OF BYTE; ix: CARDINAL; y-: ARRAY OF BYTE );
  VAR
    iy, ly: CARDINAL;
    c: BOOLEAN;
  BEGIN
    iy := 0; ly := Len0( y ); c := FALSE;
    WHILE iy < ly DO
      Dec0( x[ix], y[iy], c );
      INC( ix ); INC( iy );
    END;
    IF c THEN Dec0( x[ix], 0, c ) END;
  END Dec;

  PROCEDURE Inc( VAR x: BYTE );
  BEGIN
    x := BYTE( TSht(x) + 1 );
  END Inc;

VAR
  l: CARDINAL;
BEGIN
  (* ADR( y ) = ADR( r ) ok *)
  IF ( ADR( y ) = ADR( z ) ) OR ( ADR( z ) = ADR( r ) ) THEN RETURN FALSE END;
  IF Len0( y ) = 0 THEN RETURN FALSE END;
  Lib2.Fill( z, 0 );
  l := Len( x );
  WHILE l > 0 DO
    DEC( l );
    WHILE Cmp( x, l, y ) >= 0 DO
      Dec( x, l, y );
      IF l >= SIZE( z ) THEN RETURN FALSE END;
      Inc( z[l] );
    END;
  END;
  RETURN Cpy( r, x );
END Div;

PROCEDURE Dvd( VAR x: ARRAY OF BYTE; y-: ARRAY OF BYTE; VAR r: ARRAY OF BYTE ): BOOLEAN;
BEGIN
  (* ADR( y ) = ADR( r ) ok *)
  RETURN Div( x, y, x, r );
END Dvd;
<* POP *>

END LC.
